package com.lrdcq.playground.ui;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.RenderEffect;
import android.graphics.RenderNode;
import android.graphics.RuntimeShader;
import android.graphics.Shader;
import android.util.Log;
import android.util.SparseArray;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.FrameLayout;

import androidx.annotation.NonNull;

import com.lrdcq.playground.R;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

public class GlassView extends FrameLayout {

    public float blur_radius = 4.0f;
    public float ref_height = 30.0f;
    public float ref_length = -35.0f;
    public float ref_border_width = 5.0f;
    public float ref_exposure = 1.6f;
    public int sample_scale = 2;

    private RuntimeShader rs;
    private RenderNode mRenderNode;
    private Bitmap backgroundBitmap;
    private final Rect tmpRect = new Rect();
    private View backgroundTarget;

    private boolean isCapturingBackground = false;
    private ViewTreeObserver.OnPreDrawListener backgroundListener;

    private float mScale;

    public GlassView(@NonNull Context context) {
        super(context);
        setWillNotDraw(false);
        mPaint.setFilterBitmap(true);

        mScale = context.getResources().getDisplayMetrics().density;
        mRenderNode = new RenderNode("GlassView");
        initShader(context);
    }

    private RuntimeShader loadShaderFromRawResource(Context context, int resourceId) {
        try {
            // 从raw资源读取shader代码
            InputStream inputStream = context.getResources().openRawResource(resourceId);
            BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));

            StringBuilder shaderCode = new StringBuilder();
            String line;

            while ((line = reader.readLine()) != null) {
                shaderCode.append(line).append("\n");
            }

            reader.close();
            inputStream.close();

            // 创建RuntimeShader
            return new RuntimeShader(shaderCode.toString());

        } catch (IOException | IllegalArgumentException e) {
            return null;
        }
    }

    private void initShader(Context context) {
        rs = loadShaderFromRawResource(context, R.raw.glass);
    }

    @Override
    protected void dispatchDraw(@NonNull Canvas canvas) {
        Log.d("GlassView", "dispatchDraw");

        canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR);

        int sampledWidth = getWidth() / sample_scale;
        int sampledHeight = getHeight() / sample_scale;

        Bitmap offscreenBitmap = mBitmapPool.getBitmap(sampledWidth, sampledHeight);
        Canvas offscreenCanvas = new Canvas(offscreenBitmap);

        // 清除并绘制子视图
        offscreenCanvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR);
        offscreenCanvas.save();
        offscreenCanvas.scale(1f / sample_scale, 1f / sample_scale);
        super.dispatchDraw(offscreenCanvas);
        offscreenCanvas.restore();

        // 2. 生成SDF
        generateSDF(offscreenBitmap);

        // 4. 回收临时Bitmap
        mBitmapPool.recycle(offscreenBitmap);

        Shader background = new BitmapShader(backgroundBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
        Shader sdf = new BitmapShader(mSDFBitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
        rs.setInputShader("sdf", sdf);
        rs.setInputShader("background", background);
        rs.setFloatUniform("size", getWidth(), getHeight());
        rs.setFloatUniform("sample", (float) sample_scale);
        rs.setFloatUniform("scale", mScale);


        rs.setFloatUniform("ref_height", ref_height);
        rs.setFloatUniform("ref_length", ref_length);
        rs.setFloatUniform("ref_border_width", ref_border_width);
        rs.setFloatUniform("ref_exposure", ref_exposure);


        mRenderNode.setPosition(0, 0, backgroundBitmap.getWidth(), backgroundBitmap.getHeight());

        Canvas renderCanvas = mRenderNode.beginRecording();
        renderCanvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR);
        renderCanvas.drawBitmap(backgroundBitmap, 0, 0, new Paint());
        mRenderNode.endRecording();

        mRenderNode.setRenderEffect(
                RenderEffect.createChainEffect(
                        RenderEffect.createRuntimeShaderEffect(rs, "source"),
                        RenderEffect.createBlurEffect(blur_radius, blur_radius,Shader.TileMode.CLAMP)
                )
        );

        canvas.drawRenderNode(mRenderNode);

        //setRenderEffect(RenderEffect.createShaderEffect(rs));
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        if (w > 0 && h > 0) {
            backgroundBitmap = Bitmap.createBitmap(w, h, Bitmap.Config.ARGB_8888);
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        findBackgroundTarget();
        setupBackgroundListener();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        removeBackgroundListener();
        setRenderEffect(null);
    }

    private void findBackgroundTarget() {
        // 找到第一个非FrameLayout的父视图作为背景目标
        ViewGroup parent = (ViewGroup) getParent();

        int childCount = parent.getChildCount();
        for (int i = 0; i < childCount; i++) {
            View sibling = parent.getChildAt(i);
            if (sibling != this) {
                backgroundTarget = sibling;
                return;
            }
        }
    }

    private void setupBackgroundListener() {
        if (backgroundTarget == null) return;

        backgroundListener = () -> {
            if (backgroundTarget.isDirty() && backgroundBitmap != null && !isCapturingBackground) {
                captureBackground();
            }
            return true;
        };

        backgroundTarget.getViewTreeObserver().addOnPreDrawListener(backgroundListener);
    }

    private void captureBackground() {
        if (backgroundBitmap == null || backgroundTarget == null) return;

        isCapturingBackground = true;
        try {
            Log.d("GlassView", "captureBackground");
            tmpRect.set(0, 0, getWidth(), getHeight());
            ViewGroupUtils.offsetRectToParentCoords((ViewGroup) getParent(), this, tmpRect);

            Canvas bgCanvas = new Canvas(backgroundBitmap);

            bgCanvas.translate(-tmpRect.left, -tmpRect.top);
            backgroundTarget.draw(bgCanvas);

        } finally {
            isCapturingBackground = false;
        }
    }


    private void removeBackgroundListener() {
        if (backgroundTarget != null && backgroundListener != null) {
            backgroundTarget.getViewTreeObserver().removeOnPreDrawListener(backgroundListener);
        }
    }

    private static class ViewGroupUtils {
        static void offsetRectToParentCoords(ViewGroup parent, View child, Rect rect) {
            float offsetX = 0;
            float offsetY = 0;

            View current = child;
            while (current != null && current != parent) {
                offsetX += current.getLeft() - current.getScrollX();
                offsetY += current.getTop() - current.getScrollY();
                current = (View) current.getParent();
            }

            rect.offset((int) offsetX, (int) offsetY);
        }
    }
    private final Paint mPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
    private Bitmap mSDFBitmap;
    private static int MAX_DIST = 50;
    private static int MAX_DIST_SQ = MAX_DIST * MAX_DIST;

    // 复用数据结构
    private final BitmapPool mBitmapPool = new BitmapPool();
    private final GridDataPool mGridPool = new GridDataPool();

    private void generateSDF(Bitmap input) {
        MAX_DIST = (int) (50);
        MAX_DIST_SQ = MAX_DIST * MAX_DIST;
        final int w = input.getWidth();
        final int h = input.getHeight();

        // 从池中获取网格数据
        GridData gridData = mGridPool.getGridData(w, h);
        int[][] distSq = gridData.distSq;
        int[][] gridDx = gridData.gridDx;
        int[][] gridDy = gridData.gridDy;

        // 1. 读取像素数据
        int[] pixels = new int[w * h];
        input.getPixels(pixels, 0, w, 0, 0, w, h);

        // 2. 初始化网格 - 修正内外判断
        for (int y = 0; y < h; y++) {
            for (int x = 0; x < w; x++) {
                int alpha = (pixels[y * w + x] >> 24) & 0xFF;
                if (alpha < 0.5) { // 透明像素（边界）
                    gridDx[y][x] = 0;
                    gridDy[y][x] = 0;
                    distSq[y][x] = 0;
                } else { // 非透明像素（内部）
                    gridDx[y][x] = Integer.MAX_VALUE;
                    gridDy[y][x] = Integer.MAX_VALUE;
                    distSq[y][x] = Integer.MAX_VALUE;
                }
            }
        }

        // 3. 8SSEDT算法 - 第一遍（左上→右下）
        for (int y = 0; y < h; y++) {
            for (int x = 0; x < w; x++) {
                compare(gridDx, gridDy, distSq, x, y, -1, -1); // 左上
                compare(gridDx, gridDy, distSq, x, y, 0, -1);   // 上
                compare(gridDx, gridDy, distSq, x, y, 1, -1);   // 右上
                compare(gridDx, gridDy, distSq, x, y, -1, 0);   // 左
            }
        }

        // 4. 8SSEDT算法 - 第二遍（右下→左上）
        for (int y = h - 1; y >= 0; y--) {
            for (int x = w - 1; x >= 0; x--) {
                compare(gridDx, gridDy, distSq, x, y, 1, 0);    // 右
                compare(gridDx, gridDy, distSq, x, y, -1, 1);   // 左下
                compare(gridDx, gridDy, distSq, x, y, 0, 1);    // 下
                compare(gridDx, gridDy, distSq, x, y, 1, 1);    // 右下
            }
        }

        // 5. 获取或创建SDF Bitmap
        if (mSDFBitmap == null ||
                mSDFBitmap.getWidth() != w ||
                mSDFBitmap.getHeight() != h) {
            if (mSDFBitmap != null) {
                mBitmapPool.recycle(mSDFBitmap);
            }
            mSDFBitmap = mBitmapPool.getBitmap(w, h);
        }

        // 6. 填充SDF Bitmap - 保证所有内部像素都有有效向量
        for (int y = 0; y < h; y++) {
            for (int x = 0; x < w; x++) {
                int distSqVal = distSq[y][x];
                float r; // 距离值

                // 计算归一化距离
                if (distSqVal == 0) {
                    r = 1.0f; // 边界点
                } else if (distSqVal >= MAX_DIST_SQ) {
                    r = 0.0f; // 超出最大距离
                } else {
                    r = 1.0f - (float) Math.sqrt(distSqVal) / MAX_DIST;
                }

                // 计算向量 - 确保所有内部点都有有效向量
                float g = 0.5f;
                float b = 0.5f;

                if (distSqVal > 0 && distSqVal < Integer.MAX_VALUE) {
                    int dx = gridDx[y][x];
                    int dy = gridDy[y][x];

                    // 所有非零向量都进行计算（包括短向量）
                    if (dx != 0 || dy != 0) {
                        double len = Math.sqrt(dx * dx + dy * dy);
                        g = (float) (dx / len + 1) * 0.5f;
                        b = (float) (dy / len + 1) * 0.5f;
                    }
                }

                int color = Color.argb(255,
                        (int) (r * 255),    // R通道 = 距离
                        (int) (g * 255),    // G通道 = X方向
                        (int) (b * 255)     // B通道 = Y方向
                );
                mSDFBitmap.setPixel(x, y, color);
            }
        }

        // 7. 将网格数据返回到池中
        mGridPool.recycle(gridData);
    }

    private void compare(int[][] gridDx, int[][] gridDy, int[][] distSq,
                         int x, int y, int dx, int dy) {
        int w = gridDx[0].length;
        int h = gridDx.length;
        int nx = x + dx;
        int ny = y + dy;

        if (nx < 0 || ny < 0 || nx >= w || ny >= h) return;
        if (distSq[ny][nx] == Integer.MAX_VALUE) return;

        int newDx = gridDx[ny][nx] + dx;
        int newDy = gridDy[ny][nx] + dy;
        int newDistSq = newDx * newDx + newDy * newDy;

        if (newDistSq < distSq[y][x]) {
            distSq[y][x] = newDistSq;
            gridDx[y][x] = newDx;
            gridDy[y][x] = newDy;
        }
    }

    // 网格数据池
    private static class GridData {
        int[][] distSq;
        int[][] gridDx;
        int[][] gridDy;
        int width;
        int height;

        GridData(int w, int h) {
            width = w;
            height = h;
            distSq = new int[h][w];
            gridDx = new int[h][w];
            gridDy = new int[h][w];
        }

        boolean canReuse(int w, int h) {
            return w <= width && h <= height;
        }
    }

    private static class GridDataPool {
        private final SparseArray<GridData> pool = new SparseArray<>();

        GridData getGridData(int w, int h) {
            int sizeKey = w * 10000 + h; // 简单的尺寸键

            // 查找可复用的网格
            for (int i = 0; i < pool.size(); i++) {
                GridData data = pool.valueAt(i);
                if (data.canReuse(w, h)) {
                    pool.removeAt(i);
                    return data;
                }
            }

            // 创建新的网格
            return new GridData(w, h);
        }

        void recycle(GridData data) {
            if (data == null) return;
            int sizeKey = data.width * 10000 + data.height;
            pool.put(sizeKey, data);
        }
    }

    // Bitmap对象池
    private static class BitmapPool {
        private final SparseArray<Bitmap> pool = new SparseArray<>();

        Bitmap getBitmap(int width, int height) {
            int sizeKey = width * 10000 + height; // 简单的尺寸键

            // 查找匹配尺寸的Bitmap
            Bitmap bitmap = pool.get(sizeKey);
            if (bitmap != null) {
                pool.remove(sizeKey);
                return bitmap;
            }

            // 创建新Bitmap
            return Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        }

        void recycle(Bitmap bitmap) {
            if (bitmap == null || bitmap.isRecycled()) return;
            int sizeKey = bitmap.getWidth() * 10000 + bitmap.getHeight();

            // 只保留最近3个尺寸的Bitmap
            if (pool.size() > 3) {
                Bitmap oldest = pool.valueAt(0);
                oldest.recycle();
                pool.removeAt(0);
            }

            pool.put(sizeKey, bitmap);
        }
    }
}
